﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013
*/  
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using Inet.Viewer.Data;

namespace Inet.Viewer.WinForms
{
    /// <summary>
    /// This is the user control for the GroupTree of the report and provides the methods to fill the group tree and the drilldown function 
    /// </summary>    
    [ToolboxItem(false)] 
    public partial class GroupTreeView : UserControl, IGroupView
    {
        private ReportView reportView;

        /// <summary>
        /// Constructor to initialize this control
        /// </summary>
        public GroupTreeView()
        {
            InitializeComponent();          
            this.drillToolStripMenuItem.Click += new EventHandler(drillDownToolStripMenuItem_Click);            
            this.treeView1.AfterSelect += new TreeViewEventHandler(treeView1_AfterSelect);
        }

        /// <summary>
        /// reference to the SplitContainer
        /// </summary>
        public SplitContainer SplitContainer { get; set; }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IReportView ReportView
        {
            get
            {
                return reportView;
            }
            set
            {
                if (reportView != null)
                {
                    reportView.DataChanged -= new DataChanged(reportView_DataChanged);
                }
                reportView = value as ReportView;
                if (reportView != null)
                {
                    reportView.DataChanged += new DataChanged(reportView_DataChanged);
                }
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IReportViewer ReportViewer { get; set; }

        /// <summary>
        /// This method clears the gropup tree and fills it with the data from the <see cref="GroupTreeNode"/>
        /// </summary>
        /// <param name="groupArray">the array with the GroupTreeNodes, nothing happens if null</param>
        /// <param name="ex"></param>
        private void SetGroupTree(GroupTreeNode[] groupArray, Exception ex)
        {
            if (ex != null)
            {
                reportView.ShowError(ex);
                return;
            }
            else
            {
                TreeNodeCollection rootCol = treeView1.Nodes;
                rootCol.Clear();                
                SplitContainer.Panel1Collapsed = groupArray.Length <= 1;                
                // Start the recursion to add the nodes
                int i = 1;
                do
                {
                } while (FillTreeRecursive(groupArray, rootCol, 0, ref i) != null);
                treeView1.Refresh();           
            }
            reportView.ThreadFinished();
        }

        /// <summary>
        /// This recursive method fills the GroupTreeView with the nodes accordingly
        /// </summary>
        /// <param name="groupArray">groupArray with a list of all TreeNodes</param>
        /// <param name="nodeCol">The current collection that the node will be added to</param>
        /// <param name="curLevel">the current level of this recursion</param>
        /// <param name="i">index of the groupArray</param>
        /// <returns></returns>
        private TreeNode FillTreeRecursive(GroupTreeNode[] groupArray, TreeNodeCollection nodeCol, int curLevel, ref int i)
        {
            if (i >= groupArray.Length)
            {
                // return when all items in the array were looped through
                return null;
            }
            GroupTreeNode dNode = groupArray[i];

            if (dNode.Level > curLevel)
            {
                // add to the Tree
                TreeNode tNode = new TreeNode();                
                tNode.Text = dNode.GroupName;
                tNode.Tag = dNode;
                if (dNode.RequestURL != null && dNode.RequestURL.Length != 0)
                {
                    tNode.ContextMenuStrip = contextMenuStrip1;
                }
                nodeCol.Add(tNode);
                curLevel = dNode.Level;

                i++;               
                
                // recursion until all nodes for this level are looped through
                TreeNode upNode = null;
                do
                {
                    upNode = FillTreeRecursive(groupArray, tNode.Nodes, curLevel, ref i);
                } while (upNode != null);
                return tNode;
            }           
            return null;               
        }

        /// <summary>
        /// If a TreeView should be rendered this method clears the tree, fills the treeView
        /// and adds the events for the context menu. The internal ReportView is used.
        /// Is using the GroupLoader for this
        /// </summary>        
        public void SetupTreeView()
        {
            if (this.ReportView != null)
            {            
                ThreadManager.RequestGroupTree(this, reportView.ReportDataCache, SetGroupTree);
            }                      
        }

        /// <summary>
        /// Method invoked with click on "Drilldown" in the context menu of the tree
        /// Looks according to the Location of the contextMenuStrip from which NodeTree this 
        /// Clones the ReporRenderData and creates a new view
        /// Context menu was invoked.        
        /// </summary>
        /// <param name="sender">source of the event</param>
        /// <param name="e">the event arguments</param>
        private void drillDownToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ToolStripMenuItem toolStripItem = sender as ToolStripMenuItem;
            ToolStrip toolStrip = toolStripItem.GetCurrentParent();
            ContextMenuStrip context = toolStrip as ContextMenuStrip;
            if (context != null)
            {
                Rectangle rect = context.Bounds;
                TreeView tree = context.SourceControl as TreeView;
                // get the correct Tree Node
                Point contextPoint = new Point(rect.X, rect.Y);
                TreeNode treeNodeAt = tree.GetNodeAt(tree.PointToClient(contextPoint));
                if (this.ReportView != null && treeNodeAt != null)
                {
                    // set filter param for the subView
                    GroupTreeNode nodeInfo = treeNodeAt.Tag as GroupTreeNode;
                    if (nodeInfo != null)
                    {
                        IRenderData data = GroupLoader.GetDrillDownRenderDataCopy(nodeInfo, this.ReportView.ReportData);
                        IReportView subReport = ReportView.ReportViewer.AddNewReportView(data);
                        subReport.IsCloseable = true;
                        subReport.ReportTitle = nodeInfo.GroupName;
                        subReport.CurrentPage = 1;
                        //if (subView.PromptsOnRefresh)
                        {
                            // remove promptsonrefresh as this is deactivated for drilldown
                            //subView.ReportData[URLPARAMETER_PROMPT_ON_REFRESH, "");
                        }
                    }
                }
            }
        }

        /// <summary>
        /// This method updates the View with the new page number
        /// </summary>
        /// <param name="sender">source of the event</param>
        /// <param name="e">the event arguments</param>
        private void treeView1_AfterSelect(object sender, TreeViewEventArgs e)
        {
            if (ReportView != null)
            {
                if (this.treeView1 != null && e.Node != null)
                {
                    GroupTreeNode nodeInfo = e.Node.Tag as GroupTreeNode;
                    TreeNode testNode = this.treeView1.SelectedNode;
                    if (nodeInfo != null)
                    {
                        reportView.ContentView.ScrollTo(nodeInfo.PageFrom, new Point(0, (int)nodeInfo.YFrom), false);
                        reportView.ContentView.SelectGroup(nodeInfo);
                    }
                }
            }
        }

        /// <summary>
        /// Update the TreeView when the data of the ReportView was changed.
        /// </summary>
        /// <param name="sender">source of the event</param>
        /// <param name="timestamp">the timesstamp in millisecond</param>
        private void reportView_DataChanged(object sender, long timestamp)
        {
            this.SetupTreeView();
        }
    }
}
